#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <errno.h>

#include "imglib.h"
#include "simlib.h"

void make_model(double *para, Image *img) 
{
  int i, j;
  int npara = 6;
  static double para_pre[6] = {0., 0., 0., 0., 0., 0.};
  static int init = 1;
  static Image psf;
  double x, y;

  if(init == 1) {
    init_image(NX_PSF_FINE, NX_PSF_FINE, &psf);
  }

  init = 0;
  for(i = 2; i < npara; i++) {
    if(fabs(para[i] - para_pre[i]) > 1e-10) {
      init = 2;
    }
  }

  if(init == 2) cal_psf(para+2, &psf);

  for(i = 0; i < npara; i++)  para_pre[i] = para[i];

  clear_image(img);

  x = img->nx * 0.5 + para[0];
  y = img->ny * 0.5 + para[1];

  add(x, y, img, &psf);

  return;
}


void add(double x, double y, Image *obs, Image *psf)
{
  int i, j;
  int i2, j2;
  int icen, jcen;
  int flag = 0;
  double f = FINE_FACTOR;
  double flux_20 = 10000;
  double flux;

  icen = NX_PSF_FINE / 2;
  jcen = NY_PSF_FINE / 2;

  for(i = 0; i < NX_PSF_FINE; i++) {
    i2 = (int) ((i - icen) / f + x);
    for(j = 0; j < NY_PSF_FINE; j++) {
      j2 = (int) ((j - jcen) / f + y);

      //      printf("%d %d -> %d %d %f %f\n", i, j, i2, j2, x, y);

      if(i2 >= 0 && i2 < obs->nx && j2 >= 0 && j2 < obs->ny) {
	obs->image[i2][j2] += psf->image[i][j];
	flag = 1;
      }
    }
  }

  return;
}


void add_mag(double x, double y, double mag, Image *obs, Image *psf)
{
  int i, j;
  int i2, j2;
  int icen, jcen;
  int flag = 0;
  double f = FINE_FACTOR;
  double flux_20 = 10000;
  double flux;

  icen = NX_PSF_FINE / 2;
  jcen = NY_PSF_FINE / 2;

  flux = flux_20 * pow(10., 0.4 * (20. - mag));


  for(i = 0; i < NX_PSF_FINE; i++) {
    i2 = (int) ((i - icen) / f + x);
    for(j = 0; j < NY_PSF_FINE; j++) {
      j2 = (int) ((j - jcen) / f + y);

      //      printf("%d %d -> %d %d %f %f\n", i, j, i2, j2, x, y);

      if(i2 >= 0 && i2 < obs->nx && j2 >= 0 && j2 < obs->ny) {
	obs->image[i2][j2] += flux * psf->image[i][j];
	flag = 1;
      }
    }
  }

  return;
}


void cal_psf(double *para, Image *p)
{
  int ix, iy;
  int cen_pix_x, cen_pix_y;
  double dx, dy, r, j;
  double sum;
  double fine_factor = FINE_FACTOR;
  double i0;

  cen_pix_x = (int) (p->nx * 0.5);
  cen_pix_y = (int) (p->ny * 0.5);

  sum = 0;
  for(ix = 0; ix < p->nx; ix++) {
    for(iy = 0; iy < p->ny; iy++) {

      dx = ix - cen_pix_x;
      dy = iy - cen_pix_y;
      r = sqrt(dx * dx + dy * dy) / fine_factor;
      
      p->image[ix][iy] = 
	psf_2g(para[0], para[1], para[2], para[3], r);

      sum += p->image[ix][iy];
    }
  }

  i0 = p->image[cen_pix_x][cen_pix_y];

  sum = 0;
  for(ix = 0; ix < p->nx; ix++) {
    for(iy = 0; iy < p->ny; iy++) {
      r = fabs(ix - cen_pix_x) / fine_factor;
      j = fabs(iy - cen_pix_y) / fine_factor;
      p->image[ix][iy] += diff_spike(para, i0, r, j);
      
      r = fabs(iy - cen_pix_y) / fine_factor;
      j = fabs(ix - cen_pix_x) / fine_factor;
      p->image[ix][iy] += diff_spike(para, i0, r, j);
      
      sum += p->image[ix][iy];
    }
  }


  for(ix = 0; ix < p->nx; ix++) {
    for(iy = 0; iy < p->ny; iy++) {
      p->image[ix][iy] /= sum;
    }
  }

  return;
}

double psf_model(double *para, double dx, double dy)
{
  double alpha = para[0];
  double beta  = para[1];
  double sg1   = para[2];
  double sg2   = para[3];
  double y;
  double r;

  r = sqrt(dx * dx + dy * dy);
  y = psf_2g(alpha, beta, sg1, sg2, r);
  
  return y;
}

double psf_2g(double alpha, 
	      double beta, 
	      double sig1, 
	      double sig2, 
	      double r)
{
  double y;
  double x1, x2;
  static double f = 0.075;

  x1 = r * r / (2 * sig1 * sig1);
  x2 = r * r / (2 * sig2 * sig2);

  y = alpha * exp(-x1) + (1 - alpha - f) * exp(-x2)
    + f * pow(1 + r*r/(beta * sig2 * sig2), - beta * 0.5);

  return y;
}

double diff_spike(double *para,
		  double i0, /* central intensity */
		  double r,  /* distance from the central start 
				along ar arm of diffraction cross */
		  double j   /* distance transverse to the arm */
		  )
{
  double sg1   = para[2];
  double sg2   = para[3];
  double alpha = para[0];
  double delta = 0.01;
  double gamma = 10.;
  double spk_amp;
  double spk_cross;
  double x1, x2;

  x1 = j * j / (2 * sg1 * sg1);
  x2 = j * j / (2 * sg2 * sg2);

  spk_amp = i0 * gamma * delta / (gamma * gamma + r * r);
  spk_cross = alpha / sg1 * exp(-x1) + (1 - alpha) / sg2 * exp(-x2);
  spk_cross /= sqrt(2 * PI);

  return spk_amp * spk_cross;
}

