#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <string.h>

#include "spline.h"
#include "astro-lib.h"


double z2d(double z)
{
  return distance_conv(0, z);
}

double d2z(double d)
{
  return distance_conv(1, d);
}

double get_comoving_distance(int flag, double x)
{
    double dis;

    dis = -1;

    switch(flag) {
    case Z:
        dis = distance_conv(0, x);
        break;
    case DC:
        break;
    case DL:
        break;
    case DA:
        break;
    case DM:
        break;

    }

    return dis;
}


double get_luminocity_distance(int flag, double x)
{
    double dc;
    double dis;

    dis = -1;

    switch(flag) {
    case Z:
        dc = get_comoving_distance(Z, x);
        dis = dc * (1 + x);
        break;
    case DC:
        break;
    case DL:
        break;
    case DA:
        break;
    case DM:
        break;

    }

    return dis;
}

double get_angular_diameter_distance(int flag, double x)
{
    double dc;
    double dis;

    dis = -1;

    switch(flag) {
    case Z:
        dc = get_comoving_distance(Z, x);
        dis = dc / (1 + x);
        break;
    case DC:
        break;
    case DL:
        break;
    case DA:
        break;
    case DM:
        break;
    }

    return dis;
}

/*
 *   z2 > z1
 */
double get_angular_diameter_distance2(double z1, double z2)
{
    double dm1, dm2;
    double dm12;

    dm1 = get_comoving_distance(Z, z1);
    dm2 = get_comoving_distance(Z, z2);

    /*
    printf("z1  = %lf\n", z1);
    printf("Dm1 = %lf\n", dm1);
    printf("z2  = %lf\n", z2);
    printf("Dm2 = %lf\n", dm2);
    */

    dm12 =(dm2 - dm1) / (1 + z2);

    return dm12;
    
}

double get_distance_modulus(int flag, double x)
{
    //double dc;
    double dl;
    double dm;
    //double dis;

    dm = -1;
    switch(flag) {
    case Z:
        dl = get_luminocity_distance(Z, x);
        dm = 5 * log10(dl * 1e6 / 10.);
        /*
        printf("z  = %lf\n", x);
        printf("Dl = %lf\n", dl);
        printf("DM = %lf\n", dm);
        */
        break;
    case DC:
        break;
    case DL:
        break;
    case DA:
        break;
    case DM:
        break;
    }

    return dm;
}

/*
 *  z:      redshift
 *  dz:     redshift range width
 *  omega:  solid anlge
 *
 */

double get_comoving_volume(double z, double dz, double omega) 
{
    double dh;
    double ez;
    double da;
    double dv;
    double z2;
    double tmp;
    
    z2 = 1 + z;

    dh = C / H0;

    tmp = OMEGA_M * pow(z2, 3.) + OMEGA_L;
    ez = sqrt(tmp);

    da = get_angular_diameter_distance(Z, z);
    
    dv = dh * z2 * z2 * da * da / ez * dz * omega;
    //printf("%lf %lf %lf\n", z, omega, dv);
    
    /*
    printf("Da = %lf\n", da);
    printf("Dh = %lf\n", dh);
    */

    return dv;
}

/**
 *   flag: 0 $B$N;~(B redshift $B$+$i(B comoving distance $B$XJQ49(B
 *         1 $B$N;~(B comoving distance $B$+$i(B redshift $B$XJQ49(B
 */
double distance_conv(int flag, double x)
{
  int i;
  double dz = 0.01;
  double hz;
  double y;
  double tmp;

  static int init = 0;
  static int num;
  static double *d;
  static double *z;
  static double *coef_z2d;
  static double *coef_d2z;

  if(! init) {
      init = 1;

      d = (double *) calloc(N_DIST_CONV, sizeof(double));
      z = (double *) calloc(N_DIST_CONV, sizeof(double));
      coef_z2d = (double *) calloc(N_DIST_CONV, sizeof(double));
      coef_d2z = (double *) calloc(N_DIST_CONV, sizeof(double));
      //
      // d[] : comoving distance
      //
      d[0] = 0;
      for(i = 0; i < N_DIST_CONV; i++) {
          z[i] = i * dz;
	  //
	  // E(z)^2 = OMEGA_M * (1 + z)^3 + OMEGA_K * (1 + z)^2 + OMEGA_L
	  //
          tmp  = pow(1 + z[i], 3.0) * OMEGA_M;
          tmp += pow(1 + z[i], 4.0) * OMEGA_R;
          tmp += OMEGA_L;
          tmp -= pow(1 + z[i], 2.0) * OMEGA_K;
	  //
	  // $B@VJ}JP0\(B z $B$G$N%O%C%V%kDj?t(B
	  //
          hz = H0 * sqrt(tmp);
	  //
	  // $B?tCM@QJ,(B
	  //
          if(i > 0) {
              d[i] = d[i - 1] + C / hz * dz;
          } else {
              d[i] = 0;
          }
          //printf("%lf %lf\n", d[i], z[i]);
      }

      num = N_DIST_CONV;
  
      maketable(d, z, coef_d2z, num);
      maketable(z, d, coef_z2d, num);
  }
  
  if(flag == 0) {
      //
      // redshift --> distance
      //
      if(x > 0) {
          y = spline(x, z, d, coef_z2d, num); 
      } else {
          y = 0;
      }
  } else {
      //
      // distance --> redshift
      //
      y = spline(x, d, z, coef_d2z, num); 
  }

  return y;
}


/**
 *   flag: 0 $B$N;~(B redshift $B$+$i(B lookback time (year) $B$XJQ49(B
 *         1 $B$N;~(B lookback time (year) $B$+$i(B redshift $B$XJQ49(B
 */
double lookback_time(int flag, double x)
{
  int i;
  double dz = 0.01;
  double hz;
  double t_h;
  double y;
  double tmp;
  static int init = 0;
  static int num;
  static double *t;
  static double *z;
  static double *coef_t2z;
  static double *coef_z2t;

  //
  //
  //
  if(! init) {
      init = 1;
      t = (double *) calloc(N_DIST_CONV, sizeof(double));
      z = (double *) calloc(N_DIST_CONV, sizeof(double));
      coef_z2t = (double *) calloc(N_DIST_CONV, sizeof(double));
      coef_t2z = (double *) calloc(N_DIST_CONV, sizeof(double));
      //
      // t[] : lookback time
      //
      t[0] = 0;
      for(i = 0; i < N_DIST_CONV; i++) {
          z[i] = i * dz;
	  //
	  // E(z)^2 = OMEGA_M * (1 + z)^3 + OMEGA_K * (1 + z)^2 + OMEGA_L
	  //
          tmp  = pow(1 + z[i], 3.0) * OMEGA_M;
          tmp += pow(1 + z[i], 4.0) * OMEGA_R;
          tmp += OMEGA_L;
          tmp -= pow(1 + z[i], 2.0) * OMEGA_K;
	  //
	  // $B@VJ}JP0\(B z $B$G$N%O%C%V%kDj?t(B [km/s/Mpc]  1Mpc=3.09e19km
	  // 1yr=365.242days=31556908.800sec
	  //
          hz = H0 * sqrt(tmp);
	  t_h = 1 / hz / 31556908.800 * 3.09e19;
	  //
	  // $B?tCM@QJ,(B
	  //
          if(i > 0) {
	      t[i] = t[i - 1] + 1.0 / (1.0 + z[i]) * t_h  * dz;
          } else {
	      t[i] = 0;
          }
	  //printf("# %lf %le\n", z[i], t[i]);
      }
      //
      //
      //
      num = N_DIST_CONV;
      maketable(t, z, coef_t2z, num);
      maketable(z, t, coef_z2t, num);
  }
  //
  //
  //
  if(flag == 0) {
      //
      // redshift --> lookback time
      //
      if(x > 0) {
          y = spline(x, z, t, coef_z2t, num); 
      } else {
          y = 0;
      }
  } else {
      //
      // lookback time --> redshift
      //
      y = spline(x, t, z, coef_t2z, num); 
  }

  return y;
}

/**
 *
 */
void xyz2radec(double x, double y, double z, double *ra, double *dec)
{
    double th, ph;

    th = acos(z);
    ph = atan2(y, x);

    *ra = ph / PI * 180;
    *dec = (90 - th / PI * 180);

    return;
}

/**
 *
 */
void radec2xyz(double ra, double dec, double *x, double *y, double *z)
{
    double th, ph;

    th = (90 - dec) / 180. * PI;
    ph = ra / 180. * PI;

    *x = sin(th) * cos(ph);
    *y = sin(th) * sin(ph);
    *z = cos(th);

    return;
}

/**
 *
 */
void xyz2tp(double x, double y, double z, double *th, double *ph)
{

    *th = acos(z) / PI * 180;
    *ph = atan2(y, x) / PI * 180;

    //printf("%lf %lf %lf --> %lf %lf\n",x, y, z, *th, *ph);
	   
    return;
}

/**
 *   @param th (deg)
 *   @param ph (deg)
 *   @param x
 *   @param y
 *   @param z
 */
void tp2xyz(double th, double ph, double *x, double *y, double *z)
{

    *x = sin(th / 180. * PI) * cos(ph / 180. * PI);
    *y = sin(th / 180. * PI) * sin(ph / 180. * PI);
    *z = cos(th / 180. * PI);
	   
    return;
}



/*
 *  $BLa$jCM(B: $BN%3Q(B (deg)
 */
double angdiff(double ra1, double dec1, double ra2, double dec2) {
    double x1, y1, z1;
    double x2, y2, z2;
    double dth;
    double tmp;

    radec2xyz(ra1, dec1, &x1, &y1, &z1);
    radec2xyz(ra2, dec2, &x2, &y2, &z2);

    tmp = x1 * x2 + y1 * y2 + z1 * z2;
    if(tmp > 1.0) tmp = 1.0;
    if(tmp < -1.0) tmp = -1.0;
    
    dth = acos(tmp) / PI * 180;

    return dth;
}


/*
 * ra      (deg)
 * dec     (deg)
 * ra_str  (hh:mm:ss.sss)
 * dec_str (dd:mm:ss.sss)
 */
void deg2sexag(double ra, double dec, char *ra_str, char *dec_str) 
{
    int hh, mm, dec_d, dec_m;
    double ss, dec_s;
    
    if(ra < 0) ra += 360;

    hh = (int) (ra / 360. * 24);
    mm = (int) ((ra / 360. * 24 - hh) * 60);
    ss = ((ra / 360. * 24 - hh) * 60 - mm) * 60;
    ss = (int) (ss * 1e4 + 0.5) / 10000.;

    if(ss >= 60) {
        ss -= 60;
        mm += 1;
        if(mm >= 60) {
            mm -= 60;
            hh += 1;
            if(hh >= 24) {
                hh -= 24;
            }
        }
    }
    

    dec_d = (int) fabs(dec);
    dec_m = (int) ((fabs(dec) - dec_d) * 60);
    dec_s =  ((fabs(dec) - dec_d) * 60 - dec_m) * 60;
    dec_s = (int) (dec_s * 1e4 + 0.5) / 10000.;

    if(dec_s >= 60) {
        dec_s -= 60;
        dec_m += 1;
        if(dec_m >= 60) {
            dec_m -= 60;
            dec_d += 1;
            if(dec_d >= 90) {
                dec_d = 90;
            }
        }
    }
    
    sprintf(ra_str,  "%02d:%02d:%07.4f", hh, mm, ss);
    if(dec < 0) 
        sprintf(dec_str, "-%02d:%02d:%07.4f", dec_d, dec_m, dec_s);
    else
        sprintf(dec_str, "+%02d:%02d:%07.4f", dec_d, dec_m, dec_s);
}

void sexag2deg(char *ra_str, char *dec_str, double *ra, double *dec)
{
    int err;
    double value[3];

    err = parse_sexagesimal(ra_str, value);
    //printf("# ra_str = %s %lf %lf %lf\n", ra_str, value[0], value[1], value[2]);

    *ra  = (value[0] + value[1] / 60. + value[2] / 60. / 60.) / 24. * 360;
    
    err = parse_sexagesimal(dec_str, value);
    //printf("# dec_str = %s %lf %lf %lf\n", dec_str, value[0], value[1], value[2]);
    
    if(dec_str[0] == '-'){
        *dec = -(fabs(value[0]) + value[1] / 60. + value[2] / 60. / 60.);
    } else {
        *dec = +(fabs(value[0]) + value[1] / 60. + value[2] / 60. / 60.);
    }

    return;
}

int parse_sexagesimal(char *str, double  value[3])
{
    int i;
    int stt, stp, len;
    char v[100];

    len = strlen(str);

    /* ..... value 1 ..... */
    stt = i = 0; 
    while(str[i] != ':' && i < len) i++; stp = i - 1;
    if(stp < stt || stp >= len - 1) return 1;
    strncpy(v, &str[stt], stp - stt + 1);
    v[stp - stt + 1] = '\0';
    value[0] = atof(v);
    
    /* ..... value 2 ..... */
    i++;
    stt = stp + 2; 
    while(str[i] != ':' && i < len) i++; stp = i - 1;
    if(stp < stt || stp >= len - 1) return 1;
    strncpy(v, &str[stt], stp - stt + 1);
    v[stp - stt + 1] = '\0';
    value[1] = atof(v);
    
    /* ..... value 3 ..... */
    i++;
    stt = stp + 2;
    while(str[i] != ':' && i < len) i++; stp = i - 1;
    if(stp < stt || stp >  len - 1) return 1;
    strncpy(v, &str[stt], stp - stt + 1);
    v[stp - stt + 1] = '\0';
    value[2] = atof(v);
    
    return 0;
}


